<?php
/**
 * Plugin Name:       Easy Lead
 * Plugin URI:        https://aulisuwais.com/wordpress-plugin/
 * Description:       Capture service-business leads and auto-open WhatsApp with a prefilled message. Malaysia-ready states & routing.
 * Version:           1.8.0
 * Author:            Aulis Uwais
 * Author URI:        https://aulisuwais.com/
 * License:           GPL-2.0-or-later
 * Text Domain:       easy-lead
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'EASY_LEAD_VERSION', '1.8.0' );
define( 'EASY_LEAD_OPT', 'easy_lead_settings' );
define( 'EASY_LEAD_CPT', 'easy_lead' );

function easy_lead_default_titles() { return array("Tuan","Puan","Dato' / Datuk","Dato' Seri / Datuk Seri","Datin / Datin Seri","Tan Sri","Tun","Yang Berhormat (YB)"); }
function easy_lead_states() { return array('Johor','Kedah','Kelantan','Melaka','Negeri Sembilan','Pahang','Pulau Pinang','Perak','Perlis','Selangor','Terengganu','Sabah','Sarawak','Kuala Lumpur','Putrajaya','Labuan'); }

add_action( 'init', function(){
	register_post_type( EASY_LEAD_CPT, array(
		'labels' => array( 'name' => __( 'Leads', 'easy-lead' ), 'singular_name' => __( 'Lead', 'easy-lead' ), 'menu_name' => __( 'Easy Lead', 'easy-lead' ) ),
		'public' => false, 'show_ui' => true, 'show_in_menu' => false, 'supports' => array( 'title' ),
	) );
});

add_action( 'plugins_loaded', function(){
	add_action( 'admin_menu', 'easy_lead_admin_menu' );
	add_action( 'admin_init', 'easy_lead_register_settings' );
	add_shortcode( 'easy_lead_form', 'easy_lead_shortcode_form' );
	add_action( 'wp_ajax_easy_lead_submit', 'easy_lead_handle_submit' );
	add_action( 'wp_ajax_nopriv_easy_lead_submit', 'easy_lead_handle_submit' );
	add_action( 'admin_post_easy_lead_export', 'easy_lead_export_csv' );

	// Admin columns for All Leads
	add_filter( 'manage_edit-' . EASY_LEAD_CPT . '_columns', 'easy_lead_columns' );
	add_action( 'manage_' . EASY_LEAD_CPT . '_posts_custom_column', 'easy_lead_columns_content', 10, 2 );
	add_filter( 'manage_edit-' . EASY_LEAD_CPT . '_sortable_columns', 'easy_lead_sortable_columns' );
	add_action( 'pre_get_posts', 'easy_lead_pre_get_posts_sorting' );
	add_action( 'admin_head', 'easy_lead_list_table_css' );
});

add_action( 'admin_print_footer_scripts', function(){
	$screen = function_exists('get_current_screen') ? get_current_screen() : null;
	if ( ! $screen ) return;
	$ids = array('easy-lead_page_easy-lead-settings');
	if ( ! in_array( $screen->id, $ids, true ) ) return;
	echo '<script>(function(){try{var wrap=document.querySelector(".wrap.el-admin-wrap"),root=document.querySelector(".wrap");if(!wrap||!root) return;var header=wrap.querySelector(".el-admin-header");if(!header) return;var notices=[].slice.call(root.querySelectorAll("> .notice, > .update-nag"));notices.forEach(function(n){root.insertBefore(n, wrap);});}catch(e){}})();</script>';
});

add_action( 'admin_print_footer_scripts', function(){
	$screen = function_exists('get_current_screen') ? get_current_screen() : null;
	if ( ! $screen ) return;
	if ( $screen->id !== 'easy-lead_page_easy-lead-settings' ) return;
	echo '<script>(function(){try{var topWrap=document.getElementById("el-notices-top");if(!topWrap) return;function move(){var from=document.querySelectorAll(".el-admin-wrap .notice, .el-admin-wrap .update-nag");for(var i=0;i<from.length;i++){var n=from[i];topWrap.appendChild(n);} } move(); var obs=new MutationObserver(function(){move();}); obs.observe(document.body,{childList:true,subtree:true});}catch(e){}})();</script>';
});
/** Settings link in Plugins list + meta */
add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), function( $links ){
	$links[] = '<a href="' . esc_url( admin_url('admin.php?page=easy-lead-settings') ) . '">Settings</a>';
	return $links;
});
add_filter( 'plugin_row_meta', function( $links, $file ){
	if ( plugin_basename(__FILE__) === $file ) {
		$links[] = '<a target="_blank" href="https://aulisuwais.com/wordpress-plugin/">Plugin site</a>';
		$links[] = '<span>By Aulis Uwais</span>';
	}
	return $links;
}, 10, 2);

function easy_lead_admin_menu() {
	add_menu_page( __( 'Easy Lead', 'easy-lead' ), __( 'Easy Lead', 'easy-lead' ), 'manage_options', 'easy-lead', 'easy_lead_render_leads_page', 'dashicons-phone', 26 );
	add_submenu_page( 'easy-lead', __( 'All Leads', 'easy-lead' ), __( 'All Leads', 'easy-lead' ), 'manage_options', 'edit.php?post_type=' . EASY_LEAD_CPT );
	add_submenu_page( 'easy-lead', __( 'Settings', 'easy-lead' ), __( 'Settings', 'easy-lead' ), 'manage_options', 'easy-lead-settings', 'easy_lead_render_settings_page' );
}

function easy_lead_default_options(){
	return array(
		'business_name'   => '',
		'whatsapp_number' => '',
		'services'        => array('Aircond Service','Plumber','Electrician','Property Agent','Insurance Agent'),
		'titles'          => easy_lead_default_titles(),
		'labels'          => array(),
		'required'        => array('title'=>0,'name'=>1,'phone'=>1,'service'=>0,'city'=>1,'state'=>1,'message'=>0,'consent'=>1),
		'state_numbers'   => array(),
		'thankyou_message'=> 'Hi {name}, thanks for contacting {business}. We received your request for {service}.',
		'auto_redirect'   => 0,
		'send_email_copy' => 1,
		'email_to'        => '',
	);
}

function easy_lead_register_settings() {
	register_setting( 'easy_lead_group', EASY_LEAD_OPT, array(
		'type' => 'array',
		'sanitize_callback' => 'easy_lead_sanitize_settings',
		'default' => easy_lead_default_options(),
	) );
}

/** Merge-aware sanitize */
function easy_lead_sanitize_settings( $input ) {
	$prev = get_option( EASY_LEAD_OPT, easy_lead_default_options() );
	$out  = $prev; if ( ! is_array( $input ) ) { $input = array(); }

	if ( array_key_exists('business_name', $input) ) { $out['business_name'] = sanitize_text_field( $input['business_name'] ); }
	if ( array_key_exists('whatsapp_number', $input) ) { $out['whatsapp_number'] = preg_replace( '/\D+/', '', $input['whatsapp_number'] ); }
	if ( array_key_exists('services', $input) ) { $out['services'] = array(); $lines = is_array( $input['services'] ) ? $input['services'] : preg_split('/\r\n|\r|\n/', (string) $input['services'] ); if ( is_array( $lines ) ) { foreach ( $lines as $l ) { $l = trim( wp_strip_all_tags( $l ) ); if ( $l !== '' ) $out['services'][] = $l; } } }
	if ( array_key_exists('titles', $input) ) { $out['titles'] = array(); $tlines = is_array( $input['titles'] ) ? $input['titles'] : preg_split('/\r\n|\r|\n/', (string) $input['titles'] ); if ( is_array( $tlines ) ) { foreach ( $tlines as $l ) { $l = trim( wp_strip_all_tags( $l ) ); if ( $l !== '' ) $out['titles'][] = $l; } } if ( empty( $out['titles'] ) ) { $out['titles'] = easy_lead_default_titles(); } }
	if ( array_key_exists('labels', $input) ) { $defaults = array('title_label'=>'Title','name_label'=>'Name','phone_label'=>'Phone (WhatsApp)','service_label'=>'Service Needed','city_label'=>'City','state_label'=>'State','message_label'=>'Message','consent_label'=>'I agree to be contacted via WhatsApp/Phone.','submit_label'=>'Send via WhatsApp'); $out['labels'] = isset($out['labels']) && is_array($out['labels']) ? $out['labels'] : array(); $in_labels = is_array( $input['labels'] ) ? $input['labels'] : array(); foreach ( $defaults as $k=>$def ) { if ( array_key_exists($k, $in_labels) ) { $out['labels'][$k] = $in_labels[$k] !== '' ? sanitize_text_field( $in_labels[$k] ) : $def; } } }
	if ( array_key_exists('required', $input) ) { $req_defaults = array('title'=>0,'name'=>1,'phone'=>1,'service'=>0,'city'=>1,'state'=>1,'message'=>0,'consent'=>1); $in_req = is_array( $input['required'] ) ? $input['required'] : array(); $out['required'] = isset($out['required']) && is_array($out['required']) ? $out['required'] : $req_defaults; foreach ( $req_defaults as $k=>$def ) { $out['required'][$k] = ! empty( $in_req[$k] ) ? 1 : 0; } }
	if ( array_key_exists('state_numbers', $input) ) { $out['state_numbers'] = array(); if ( is_array( $input['state_numbers'] ) ) { foreach ( $input['state_numbers'] as $state => $num ) { $state = sanitize_text_field( $state ); $number = preg_replace('/\D+/', '', $num ); if ( $state !== '' && $number !== '' ) { $out['state_numbers'][$state] = $number; } } } }
	if ( array_key_exists('thankyou_message', $input) ) { $out['thankyou_message'] = wp_kses_post( $input['thankyou_message'] ); }
	if ( array_key_exists('auto_redirect', $input) ) { $out['auto_redirect'] = ! empty( $input['auto_redirect'] ) ? 1 : 0; }
	if ( array_key_exists('send_email_copy', $input) ) { $out['send_email_copy'] = ! empty( $input['send_email_copy'] ) ? 1 : 0; }
	if ( array_key_exists('email_to', $input) ) { $out['email_to'] = sanitize_email( $input['email_to'] ); }

	return $out;
}

/** Admin: All Leads page container + CSV */
function easy_lead_render_leads_page() {
	if ( ! current_user_can( 'manage_options' ) ) return;
	$export_url = wp_nonce_url( admin_url( 'admin-post.php?action=easy_lead_export' ), 'easy_lead_export' );
	echo '<div class="wrap"><h1>Easy Lead <span style="font-size:12px;background:#111827;color:#fff;padding:3px 8px;border-radius:999px;">v' . esc_html( EASY_LEAD_VERSION ) . '</span></h1>';
	echo '<p><a class="button button-primary" href="' . esc_url( $export_url ) . '">Export CSV</a> ';
	echo '<a class="button" href="' . esc_url( admin_url( 'admin.php?page=easy-lead-settings' ) ) . '">Settings</a> ';
	echo '<a class="button" target="_blank" href="https://aulisuwais.com/wordpress-plugin/">Visit plugin site</a></p>';
	echo '</div>';
}

/** Admin Columns */
function easy_lead_columns( $columns ) {
	$new = array();
	foreach ( $columns as $key => $label ) {
		if ( $key === 'cb' || $key === 'title' || $key === 'date' ) { $new[$key] = $label; }
	}
	$ordered = array();
	foreach ( $new as $key => $label ) {
		$ordered[$key] = $label;
		if ( $key === 'title' ) {
			$ordered['el_name']    = __( 'Name', 'easy-lead' );
			$ordered['el_phone']   = __( 'Phone', 'easy-lead' );
			$ordered['el_service'] = __( 'Service', 'easy-lead' );
			$ordered['el_city']    = __( 'City', 'easy-lead' );
			$ordered['el_state']   = __( 'State', 'easy-lead' );
		}
	}
	return $ordered;
}

function easy_lead_columns_content( $column, $post_id ) {
	switch ( $column ) {
		case 'el_name':
			echo esc_html( get_post_field( 'post_title', $post_id ) );
			break;
		case 'el_phone':
			$phone = get_post_meta( $post_id, 'phone', true );
			$num   = preg_replace( '/\D+/', '', $phone );
			if ( $num ) {
				echo '<a href="' . esc_url( 'https://wa.me/' . rawurlencode( $num ) ) . '" target="_blank" title="Open WhatsApp">' . esc_html( $phone ) . '</a>';
			} else {
				echo '&mdash;';
			}
			break;
		case 'el_service':
			$service = get_post_meta( $post_id, 'service', true );
			echo $service ? esc_html( $service ) : '&mdash;';
			break;
		case 'el_city':
			$city = get_post_meta( $post_id, 'city', true );
			echo $city ? esc_html( $city ) : '&mdash;';
			break;
		case 'el_state':
			$state = get_post_meta( $post_id, 'state', true );
			echo $state ? esc_html( $state ) : '&mdash;';
			break;
	}
}

function easy_lead_sortable_columns( $columns ) { return $columns; }
function easy_lead_pre_get_posts_sorting( $query ) {
	if ( is_admin() && $query->is_main_query() && $query->get('post_type') === EASY_LEAD_CPT ) { /* Keep default date DESC */ }
}

/** Slightly wider, readable list table */
function easy_lead_list_table_css() {
	$screen = get_current_screen();
	if ( ! $screen || $screen->id !== 'edit-' . EASY_LEAD_CPT ) return;
	echo '<style>
		.wp-list-table.widefat.fixed { table-layout: auto; }
		.column-el_phone { width: 140px; }
		.column-el_service { width: 180px; }
		.column-el_city { width: 140px; }
		.column-el_state { width: 140px; }
	</style>';
}

/** ===== FRONTEND + Settings UI ===== */

function easy_lead_get_label( $key, $default ) { $opts = get_option( EASY_LEAD_OPT, array() ); return isset( $opts['labels'][$key] ) && $opts['labels'][$key] !== '' ? esc_html( $opts['labels'][$key] ) : esc_html( $default ); }
function easy_lead_is_required( $key ) { $opts = get_option( EASY_LEAD_OPT, array() ); return ( isset($opts['required'][$key]) && $opts['required'][$key] ) ? 'required' : ''; }

function easy_lead_shortcode_form( $atts ) {
	$opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() );
	$services = ( isset($opts['services']) && is_array($opts['services']) && count($opts['services']) ) ? $opts['services'] : array('Aircond Service','Plumber','Electrician','Property Agent','Insurance Agent');
	$titles   = ( isset($opts['titles']) && is_array($opts['titles']) && count($opts['titles']) ) ? $opts['titles'] : easy_lead_default_titles();

	$state_opts=''; foreach ( easy_lead_states() as $st ) { $state_opts .= '<option value="' . esc_attr($st) . '">' . esc_html($st) . '</option>'; }
	$title_opts=''; foreach ( $titles as $t ) { $title_opts .= '<option value="' . esc_attr($t) . '">' . esc_html($t) . '</option>'; }
	$services_opts=''; foreach ( $services as $s ) { $services_opts .= '<option value="' . esc_attr( $s ) . '">' . esc_html( $s ) . '</option>'; }

	$config = array(
		'ajax_url'      => admin_url('admin-ajax.php'),
		'nonce'         => wp_create_nonce('easy_lead_nonce'),
		'auto_redirect' => ! empty( $opts['auto_redirect'] ) ? true : false,
		'whatsapp'      => isset($opts['whatsapp_number']) ? preg_replace('/\D+/', '', $opts['whatsapp_number']) : '',
		'business'      => isset($opts['business_name']) ? $opts['business_name'] : get_bloginfo('name'),
		'template'      => isset($opts['thankyou_message']) ? $opts['thankyou_message'] : '',
		'state_numbers' => isset($opts['state_numbers']) && is_array($opts['state_numbers']) ? $opts['state_numbers'] : array(),
	);

	ob_start(); ?>
	<style><?php echo easy_lead_inline_css(); ?></style>
	<form class="easy-lead-form el-modern" method="post" action="#">
		<?php echo wp_nonce_field( 'easy_lead_submit', '_wpnonce', true, false ); ?>
		<input type="hidden" name="action" value="easy_lead_submit">
		<div class="el-grid">
			<label class="el-field"><span class="el-label"><?php echo easy_lead_get_label('title_label','Title'); ?></span><select name="title" <?php echo easy_lead_is_required('title'); ?> style="width:100%"><?php echo $title_opts; ?></select></label>
			<label class="el-field"><span class="el-label"><?php echo easy_lead_get_label('name_label','Name'); ?></span><input type="text" name="name" <?php echo easy_lead_is_required('name'); ?>></label>
			<label class="el-field"><span class="el-label"><?php echo easy_lead_get_label('phone_label','Phone (WhatsApp)'); ?></span><input type="tel" name="phone" pattern="[0-9+\s-]+" title="Digits, +, spaces or - only" <?php echo easy_lead_is_required('phone'); ?>></label>
			<label class="el-field"><span class="el-label"><?php echo easy_lead_get_label('service_label','Service Needed'); ?></span><select name="service" <?php echo easy_lead_is_required('service'); ?> style="width:100%"><?php echo $services_opts; ?></select></label>
			<label class="el-field"><span class="el-label"><?php echo easy_lead_get_label('city_label','City'); ?></span><input type="text" name="city" placeholder="e.g. Shah Alam" <?php echo easy_lead_is_required('city'); ?>></label>
			<label class="el-field"><span class="el-label"><?php echo easy_lead_get_label('state_label','State'); ?></span><select name="state" <?php echo easy_lead_is_required('state'); ?> style="width:100%"><?php echo $state_opts; ?></select></label>
			<label class="el-field el-col-span-2 el-message"><span class="el-label"><?php echo easy_lead_get_label('message_label','Message'); ?></span><textarea name="message" rows="4" placeholder="Tell us more..." <?php echo easy_lead_is_required('message'); ?>></textarea></label>

			<div class="el-consent el-col-span-2">
				<label class="el-consent-box"><input type="checkbox" name="consent" <?php echo easy_lead_is_required('consent'); ?>><span><?php echo easy_lead_get_label('consent_label','I agree to be contacted via WhatsApp/Phone.'); ?></span></label>
			</div>

			<div class="el-actions el-col-span-2"><button type="submit" class="el-btn"><?php echo easy_lead_get_label('submit_label','Send via WhatsApp'); ?></button><span class="el-spinner" style="display:none"></span><span class="el-success" style="display:none"></span><span class="el-error" style="display:none"></span></div>
		</div>
	</form>
	<script><?php echo easy_lead_inline_js( $config ); ?></script>
	<?php
	return ob_get_clean();
}

function easy_lead_inline_css() {
$css = <<<CSS
.easy-lead-form.el-modern{
  isolation:isolate; max-width:760px; margin:28px auto; padding:28px;
  border:1px solid rgba(17,24,39,.08); border-radius:24px; background:#fff;
  box-shadow:0 18px 60px rgba(0,0,0,.08);
  font-family:Inter,system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Cantarell,Noto Sans,sans-serif;
  backdrop-filter:saturate(1.2);
}
.easy-lead-form.el-modern *{box-sizing:border-box}
.el-modern .el-grid{display:grid !important; grid-template-columns:1fr 1fr; gap:18px}
.el-modern .el-field{display:flex; flex-direction:column; gap:10px}
.el-modern .el-col-span-2{grid-column:1 / -1}
.el-modern .el-label{font-weight:700; letter-spacing:.2px; color:#0f172a; font-size:14.5px}
.el-modern input,.el-modern select,.el-modern textarea{
  width:100%; padding:14px 16px; border:1px solid #e5e7eb; border-radius:14px; outline:none; background:#fff;
  transition: box-shadow .15s ease, border-color .15s ease;
}
.el-modern input:focus,.el-modern select:focus,.el-modern textarea:focus{
  border-color:#111827; box-shadow:0 0 0 4px rgba(17,24,39,.08);
}
.el-modern textarea{min-height:120px; resize:vertical}
.el-modern .el-message{grid-column:1 / -1}
.el-modern .el-consent{grid-column:1 / -1 !important; display:block !important; margin-top:6px !important; width:100% !important}
.el-modern .el-consent label.el-consent-box{
  display:flex !important; flex-wrap:nowrap !important; align-items:flex-start !important; gap:12px !important;
  margin:0 !important; width:100% !important; line-height:1.5 !important;
}
.el-modern .el-consent input[type="checkbox"]{
  all:revert; margin-top:2px !important; flex:0 0 auto !important; width:auto !important; height:auto !important;
}
.el-modern .el-consent .el-consent-box > span{display:block !important; margin:0 !important}
.tcb-col .el-modern .el-consent{width:100% !important}
.tcb-flex-row .el-modern .el-consent .el-consent-box{display:flex !important}
.el-modern .el-actions{display:flex; align-items:center; gap:14px}
.el-modern .el-btn{
  background:linear-gradient(135deg,#111827,#0b1220); color:#fff; border:none; padding:13px 22px;
  border-radius:999px; cursor:pointer; font-weight:800; letter-spacing:.2px; box-shadow:0 10px 24px rgba(17,24,39,.16);
}
.el-modern .el-btn:hover{transform:translateY(-1px)}
.el-modern .el-spinner:before{content:""; display:inline-block; width:18px; height:18px; border:2px solid #9ca3af; border-left-color:transparent; border-radius:50%; animation:elspin 1s linear infinite}
@keyframes elspin{to{transform:rotate(360deg)}}
.el-modern .el-success{color:#059669; font-weight:600}
.el-modern .el-error{color:#dc2626; font-weight:600}
@media (max-width:680px){ .el-modern .el-grid{grid-template-columns:1fr} }
CSS;
return $css . '.el-admin-wrap .notice,.el-admin-wrap .update-nag{display:none!important}';
}

function easy_lead_inline_js( $config ) {
	$json = wp_json_encode( $config );
	return "(function(){ if(window.__EasyLeadBound)return; window.__EasyLeadBound=true; var CFG={$json}; function buildWa(data){ var business=CFG.business||''; var tpl=CFG.template||'Hi {name}, thanks for contacting {business}. We received your request for {service}.'; function r(s){return tpl.replace('{title}',data.title||'').replace('{name}',data.name||'').replace('{phone}',data.phone||'').replace('{service}',data.service||'').replace('{city}',data.city||'').replace('{state}',data.state||'').replace('{message}',data.message||'').replace('{business}',business);} var pre=r(''); var details='— Lead Details —\\n'+'Title: '+(data.title||'')+'\\n'+'Name: '+(data.name||'')+'\\n'+'Phone: '+(data.phone||'')+'\\n'+'Service: '+(data.service||'')+'\\n'+'City: '+(data.city||'')+'\\n'+'State: '+(data.state||'')+'\\n'+'Message: '+(data.message||''); var text=pre+'\\n\\n'+details; var num=CFG.whatsapp||''; if (CFG.state_numbers && data.state && CFG.state_numbers[data.state]){ num=CFG.state_numbers[data.state]; } if(!num){ num=(data.phone||'').replace(/\\D+/g,''); } return 'https://wa.me/'+encodeURIComponent(num)+'?text='+encodeURIComponent(text); } function bindForm(f){ function qs(s){return f.querySelector(s);} function collect(){ return { title:(f.querySelector('[name=\"title\"]').value||''), name:(f.querySelector('[name=\"name\"]').value||''), phone:(f.querySelector('[name=\"phone\"]').value||''), service:(f.querySelector('[name=\"service\"]').value||''), city:(f.querySelector('[name=\"city\"]').value||''), state:(f.querySelector('[name=\"state\"]').value||''), message:(f.querySelector('[name=\"message\"]').value||''), consent:(f.querySelector('[name=\"consent\"]').checked?1:0) }; } function onSubmit(e){ e.preventDefault(); var btn=qs('.el-btn'),spin=qs('.el-spinner'),ok=qs('.el-success'),err=qs('.el-error'); if(ok)ok.style.display='none'; if(err)err.style.display='none'; if(spin)spin.style.display='inline-block'; if(btn)btn.disabled=true; var fd=new FormData(f); fd.append('security', CFG.nonce ); var xhr=new XMLHttpRequest(); var finished=false; xhr.open('POST', CFG.ajax_url, true); xhr.onreadystatechange=function(){ if(xhr.readyState===4){ finished=true; if(spin)spin.style.display='none'; if(btn)btn.disabled=false; try{ var r=JSON.parse(xhr.responseText); if(r && r.success && r.data && r.data.wa){ if(r.data.redirect){ window.location=r.data.wa; } else { window.open(r.data.wa,'_blank'); } } else { var wa=buildWa(collect()); if(CFG.auto_redirect){ window.location=wa; } else { window.open(wa, '_blank'); } } } catch(ex){ var wa=buildWa(collect()); if(CFG.auto_redirect){ window.location=wa; } else { window.open(wa, '_blank'); } } } }; xhr.send(fd); setTimeout(function(){ if(!finished){ if(spin)spin.style.display='none'; if(btn)btn.disabled=false; var wa=buildWa(collect()); if(CFG.auto_redirect){ window.location=wa; } else { window.open(wa, '_blank'); } } }, 5000); } if(!f.__elBound){ f.addEventListener('submit', onSubmit); f.__elBound=true; } } var forms=document.querySelectorAll('form.easy-lead-form'); for(var i=0;i<forms.length;i++){ bindForm(forms[i]); } })();";
}

/** ===== Admin Settings UI (modern) ===== */
function easy_lead_admin_css(){
$css = <<<CSS
.el-admin-wrap{max-width:1100px}
.el-admin-header{display:flex;align-items:center;justify-content:space-between;margin:10px 0 18px 0}
.el-brand{display:flex;align-items:center;gap:12px}
.el-logo{width:36px;height:36px;border-radius:10px;background:#111827;color:#fff;display:flex;align-items:center;justify-content:center;font-weight:800;letter-spacing:.5px}
.el-title{font-size:22px;font-weight:800;color:#0b1220}
.el-badge{font-size:12px;background:#111827;color:#fff;padding:3px 8px;border-radius:999px;margin-left:8px}
.el-links{display:flex;align-items:center;gap:10px;color:#374151}
.el-link{text-decoration:none}
.el-dot{width:4px;height:4px;border-radius:50%;background:#9ca3af;display:inline-block}
.el-tabs{display:flex;gap:8px;margin:0 0 14px 0;flex-wrap:wrap}
.el-tabs a{padding:8px 12px;border:1px solid #e5e7eb;border-radius:999px;text-decoration:none;color:#111827;font-weight:600;background:#fff}
.el-tabs a.active{background:#111827;color:#fff;border-color:#111827}
.el-card{background:#fff;border:1px solid #e5e7eb;border-radius:16px;padding:18px 18px 22px 18px;box-shadow:0 6px 20px rgba(0,0,0,.06)}
.el-sub{color:#4b5563;margin-top:-4px}
.el-grid-2{display:grid;grid-template-columns:1fr 1fr;gap:16px}
@media(max-width:800px){.el-grid-2{grid-template-columns:1fr}}
.el-table{display:grid;gap:0;border:1px solid #e5e7eb;border-radius:12px;overflow:hidden}
.el-tbody .el-tr{display:grid;grid-template-columns:1fr 1fr;border-top:1px solid #e5e7eb;padding:10px 12px;align-items:center}
.el-td-key{color:#111827;font-weight:600}
.el-chip{display:inline-flex;align-items:center;gap:8px;padding:8px 12px;border:1px solid #e5e7eb;border-radius:999px;background:#fff;font-weight:600}
.el-switch{display:inline-flex;align-items:center;gap:10px;padding:8px 10px;border:1px solid #e5e7eb;border-radius:12px;background:#fff}
.el-switch input[type="checkbox"]{transform:scale(1.1)}
.el-shortcodes code{background:#f3f4f6;padding:2px 6px;border-radius:6px;border:1px solid #e5e7eb;margin-right:6px;display:inline-block}
.el-notice{margin:10px 0 0 0;border-left:4px solid #10b981;background:#ecfdf5;padding:10px 12px;border-radius:6px}
CSS;
return $css . '.el-admin-wrap .notice,.el-admin-wrap .update-nag{display:none!important}';
}

function easy_lead_field_text( $args ) { $opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() ); $id = isset($args['id']) ? esc_attr( $args['id'] ) : ''; $val = isset( $opts[$id] ) && ! is_array($opts[$id]) ? esc_attr( $opts[$id] ) : ''; $ph  = isset( $args['placeholder'] ) ? esc_attr( $args['placeholder'] ) : ''; echo "<input type='text' class='regular-text' id='{$id}' name='" . EASY_LEAD_OPT . "[{$id}]' value='{$val}' placeholder='{$ph}'>"; }
function easy_lead_field_textarea( $args ) { $opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() ); $id = isset($args['id']) ? esc_attr( $args['id'] ) : ''; $val = ''; if ( isset( $opts[$id] ) ) { $val = is_array( $opts[$id] ) ? esc_textarea( implode( "\n", $opts[$id] ) ) : esc_textarea( (string) $opts[$id] ); } $ph  = isset( $args['placeholder'] ) ? esc_attr( $args['placeholder'] ) : ''; echo "<textarea style='min-height:120px' class='large-text code' rows='5' id='{$id}' name='" . EASY_LEAD_OPT . "[{$id}]' placeholder='{$ph}'>" . $val . "</textarea>"; }
function easy_lead_field_labels_group() { $opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() ); $labels = isset( $opts['labels'] ) && is_array($opts['labels']) ? $opts['labels'] : array(); $fields = array('title_label'=>'Title','name_label'=>'Name','phone_label'=>'Phone (WhatsApp)','service_label'=>'Service Needed','city_label'=>'City','state_label'=>'State','message_label'=>'Message','consent_label'=>'Consent text','submit_label'=>'Submit button'); echo "<div class='el-table'><div class='el-tbody'>"; foreach ( $fields as $key => $default ) { $val = isset( $labels[$key] ) && $labels[$key] !== '' ? $labels[$key] : $default; echo "<div class='el-tr'><div class='el-td el-td-key'>{$default}</div><div class='el-td'><input type='text' class='regular-text' name='" . EASY_LEAD_OPT . "[labels][{$key}]' value='" . esc_attr($val) . "'></div></div>"; } echo "</div></div>"; }
function easy_lead_field_required_group() { $opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() ); $req = isset( $opts['required'] ) && is_array($opts['required']) ? $opts['required'] : array(); $fields = array('title'=>'Title','name'=>'Name','phone'=>'Phone','service'=>'Service Needed','city'=>'City','state'=>'State','message'=>'Message','consent'=>'Consent'); echo "<div class='el-grid-2'>"; foreach ( $fields as $key => $label ) { $checked = ! empty( $req[$key] ) ? 'checked' : ''; echo "<label class='el-chip'><input type='checkbox' name='" . EASY_LEAD_OPT . "[required][{$key}]' value='1' {$checked}> {$label}</label>"; } echo "</div>"; }

function easy_lead_field_states_table() {
	$opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() );
	$current = isset($opts['state_numbers']) && is_array($opts['state_numbers']) ? $opts['state_numbers'] : array();
	echo '<div class="el-table"><div class="el-tbody">';
	foreach ( easy_lead_states() as $st ) {
		$val = isset($current[$st]) ? $current[$st] : '';
		echo '<div class="el-tr"><div class="el-td el-td-key">'. esc_html($st) .'</div><div class="el-td"><input type="text" class="regular-text" name="'. EASY_LEAD_OPT .'[state_numbers]['. esc_attr($st) .']" value="'. esc_attr($val) .'" placeholder="60123456789"></div></div>';
	}
	echo '</div></div>';
}

function easy_lead_render_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) return;
	$allowed_tabs = array('general','form','lists','routing','messaging','notifications','shortcode','about');
	$active = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'general';
	if ( ! in_array($active, $allowed_tabs, true) ) { $active = 'general'; }

	echo '<div class="wrap" id="el-notices-top"></div><div class="el-admin-wrap">';
	echo '<style>'. easy_lead_admin_css() .'</style>';
	echo '<div class="el-admin-header"><div class="el-brand"><div class="el-logo">EL</div><div class="el-title">Easy Lead <span class="el-badge">v'. esc_html(EASY_LEAD_VERSION) .'</span></div></div><div class="el-links"><a target="_blank" href="https://aulisuwais.com/wordpress-plugin/" class="el-link">Plugin site</a><span class="el-dot"></span><span class="el-author">By Aulis Uwais</span></div></div>';

	// Success notice after save
	if ( isset($_GET['settings-updated']) && $_GET['settings-updated'] ) {
		echo '<div class="el-notice">✅ Settings saved successfully.</div>';
	}

	// Tabs
	$tabs = array('general'=>'General','form'=>'Form','lists'=>'Lists','routing'=>'Routing','messaging'=>'Messaging','notifications'=>'Notifications','shortcode'=>'Shortcode','about'=>'About');
	echo '<div class="el-tabs">'; foreach ( $tabs as $k=>$l ){ $url = esc_url( admin_url( 'admin.php?page=easy-lead-settings&tab=' . $k ) ); $cls = $active===$k?'class="active"':''; echo "<a {$cls} href='{$url}'>{$l}</a>"; } echo '</div>';

	// Shortcode tab: only shows shortcode and exits early
	if ( $active === 'shortcode' ) {
		echo '<div class="el-card"><h2>Shortcode</h2><p class="el-sub">Paste this shortcode in a page or post.</p><code style="font-size:14px;padding:10px 12px;border:1px solid #e5e7eb;border-radius:8px;display:inline-block;background:#f9fafb">[easy_lead_form]</code></div></div>';
		return;
	}

	echo '<form method="post" action="options.php"><div class="el-card">';
	settings_fields( 'easy_lead_group' );

	switch ( $active ) {
		case 'form':
			echo '<h2>Form</h2><p class="el-sub">Edit field labels and choose which ones are required.</p>';
			echo '<h3>Labels</h3>'; easy_lead_field_labels_group();
			echo '<h3 style="margin-top:18px">Required</h3>'; easy_lead_field_required_group();
			break;
		case 'lists':
			echo '<h2>Lists</h2><div class="el-grid-2"><div><h3>Services (one per line)</h3>'; easy_lead_field_textarea( array('id'=>'services') ); echo '</div><div><h3>Titles (one per line)</h3>'; easy_lead_field_textarea( array('id'=>'titles') ); echo '</div></div>';
			break;
		case 'routing':
			echo '<h2>State-based WhatsApp Routing</h2><p class="el-sub">Assign a WhatsApp number for each state (optional). If empty, the default number is used.</p>'; easy_lead_field_states_table(); break;
		case 'messaging':
			$opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() );
			echo '<h2>WhatsApp Prefilled Template</h2><p class="el-sub">Template is added first, then full lead details.</p>'; easy_lead_field_textarea( array('id'=>'thankyou_message') );
			echo '<div class="el-shortcodes"><strong>Available variables:</strong> <code>{title}</code> <code>{name}</code> <code>{phone}</code> <code>{service}</code> <code>{city}</code> <code>{state}</code> <code>{message}</code> <code>{business}</code></div>';
			echo '<label class="el-switch"><input type="checkbox" name="'.EASY_LEAD_OPT.'[auto_redirect]" value="1" '. ( !empty($opts['auto_redirect']) ? 'checked' : '' ) .'><span></span> Auto-redirect user to WhatsApp</label>';
			break;
		case 'notifications':
			$opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() );
			echo '<h2>Email Notifications</h2><label class="el-switch"><input type="checkbox" name="'.EASY_LEAD_OPT.'[send_email_copy]" value="1" '. ( !empty($opts['send_email_copy']) ? 'checked' : '' ) .'><span></span> Send email copy to admin</label>';
			echo '<p style="margin-top:12px"><label>Admin Email<br>'; easy_lead_field_text( array('id'=>'email_to','placeholder'=>get_option('admin_email') ) ); echo '</label></p>';
			break;
		case 'about':
			echo '<h2>About</h2><p>Easy Lead helps service businesses capture leads and jump into WhatsApp fast.</p>';
			break;
		default:
			echo '<h2>General</h2><div class="el-grid-2"><p><label>Business Name<br>'; easy_lead_field_text( array('id'=>'business_name','placeholder'=>'My Company') ); echo '</label></p><p><label>Default WhatsApp Number<br>'; easy_lead_field_text( array('id'=>'whatsapp_number','placeholder'=>'60123456789') ); echo '</label></p></div>';
			break;
	}
	submit_button( __( 'Save Settings', 'easy-lead' ) );
	echo '</div></form></div>';
}

/** AJAX + Email + CSV */
function easy_lead_verify_any_nonce() {
	if ( isset($_POST['security']) && wp_verify_nonce( sanitize_text_field($_POST['security']), 'easy_lead_nonce' ) ) { return true; }
	if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( $_POST['_wpnonce'] ), 'easy_lead_submit' ) ) { return true; }
	return false;
}
function easy_lead_handle_submit() {
	if ( ! easy_lead_verify_any_nonce() ) { wp_send_json_error( array( 'message' => 'Security validation failed.' ) ); }
	$data = array(
		'title'   => isset($_POST['title']) ? sanitize_text_field( $_POST['title'] ) : '',
		'name'    => isset($_POST['name']) ? sanitize_text_field( $_POST['name'] ) : '',
		'phone'   => isset($_POST['phone']) ? preg_replace( '/\D+/', '', $_POST['phone'] ) : '',
		'service' => isset($_POST['service']) ? sanitize_text_field( $_POST['service'] ) : '',
		'city'    => isset($_POST['city']) ? sanitize_text_field( $_POST['city'] ) : '',
		'state'   => isset($_POST['state']) ? sanitize_text_field( $_POST['state'] ) : '',
		'message' => isset($_POST['message']) ? sanitize_textarea_field( $_POST['message'] ) : '',
		'consent' => isset($_POST['consent']) ? 1 : 0,
	);
	$opts = get_option( EASY_LEAD_OPT, easy_lead_default_options() );
	$req  = isset($opts['required']) && is_array($opts['required']) ? $opts['required'] : array();
	$missing = array(); foreach ( array('title','name','phone','service','city','state','message','consent') as $k ) { if ( ! empty( $req[$k] ) && empty( $data[$k] ) ) { $missing[] = ucfirst($k); } }
	if ( ! empty($missing) ) { wp_send_json_error( array( 'message' => 'Please fill required fields: ' . implode(', ', $missing) ) ); }

	$post_id = wp_insert_post( array( 'post_type' => EASY_LEAD_CPT, 'post_status' => 'publish', 'post_title' => trim($data['title'].' '.$data['name']) ) );
	if ( is_wp_error( $post_id ) ) { wp_send_json_error( array( 'message' => __( 'Could not save lead.', 'easy-lead' ) ) ); }
	foreach ( $data as $k => $v ) { update_post_meta( $post_id, $k, $v ); }

	$business = ( isset($opts['business_name']) && $opts['business_name'] ) ? $opts['business_name'] : get_bloginfo( 'name' );
	$template = ( isset($opts['thankyou_message']) && $opts['thankyou_message'] ) ? $opts['thankyou_message'] : 'Hi {name}, thanks for contacting {business}. We received your request for {service}.';
	$repl = array('{title}'=>$data['title'],'{name}'=>$data['name'],'{phone}'=>$data['phone'],'{service}'=>$data['service'],'{message}'=>$data['message'],'{city}'=>$data['city'],'{state}'=>$data['state'],'{business}'=>$business);
	$prefill_first = strtr( $template, $repl );
	$details = "— Lead Details —\n"."Title: {$data['title']}\n"."Name: {$data['name']}\n"."Phone: {$data['phone']}\n"."Service: {$data['service']}\n"."City: {$data['city']}\n"."State: {$data['state']}\n"."Message: {$data['message']}";
	$wa_text = $prefill_first . "\n\n" . $details;

	$target_number = isset($opts['whatsapp_number']) ? preg_replace('/\D+/', '', $opts['whatsapp_number']) : '';
	if ( isset($opts['state_numbers'][ $data['state'] ]) && $opts['state_numbers'][ $data['state'] ] ) { $target_number = preg_replace('/\D+/', '', $opts['state_numbers'][ $data['state'] ] ); }
	if ( ! $target_number ) { $target_number = $data['phone']; }
	$wa_url = 'https://wa.me/' . rawurlencode( $target_number ) . '?text=' . rawurlencode( $wa_text );

	if ( ! empty( $opts['send_email_copy'] ) ) {
		$to = ! empty( $opts['email_to'] ) ? $opts['email_to'] : get_option( 'admin_email' );
		$subject = '[Easy Lead] New lead from ' . trim($data['title'].' '.$data['name']);
		$body = "Title: {$data['title']}\nName: {$data['name']}\nPhone: {$data['phone']}\nService: {$data['service']}\nCity: {$data['city']}\nState: {$data['state']}\nMessage: {$data['message']}\n";
		wp_mail( $to, $subject, $body );
	}
	wp_send_json_success( array( 'redirect' => ! empty( $opts['auto_redirect'] ), 'wa' => $wa_url, 'lead_id' => $post_id ) );
}

function easy_lead_export_csv() {
	if ( ! current_user_can( 'manage_options' ) ) { wp_die( 'Forbidden.' ); }
	check_admin_referer( 'easy_lead_export' );
	header( 'Content-Type: text/csv; charset=utf-8' );
	header( 'Content-Disposition: attachment; filename=easy-leads-' . date( 'Ymd-His' ) . '.csv' );
	$out = fopen( 'php://output', 'w' );
	fputcsv( $out, array( 'ID', 'Created', 'Title', 'Name', 'Phone', 'Service', 'City', 'State', 'Message' ) );
	$q = new WP_Query( array( 'post_type' => EASY_LEAD_CPT, 'posts_per_page' => -1, 'orderby' => 'date', 'order' => 'DESC' ) );
	if ( $q && isset($q->posts) ) {
		foreach ( $q->posts as $p ) {
			fputcsv( $out, array(
				$p->ID,
				get_the_date( 'Y-m-d H:i', $p ),
				get_post_meta( $p->ID, 'title', true ),
				$p->post_title,
				get_post_meta( $p->ID, 'phone', true ),
				get_post_meta( $p->ID, 'service', true ),
				get_post_meta( $p->ID, 'city', true ),
				get_post_meta( $p->ID, 'state', true ),
				get_post_meta( $p->ID, 'message', true ),
			) );
		}
	}
	fclose( $out ); exit;
}
